# tesseract_setup.py
# Helper to configure Tesseract path for bundled or installed version

import os
import sys
import pytesseract
from pathlib import Path


def find_tesseract():
    """Find Tesseract executable in various locations"""

    # If running as PyInstaller bundle
    if getattr(sys, 'frozen', False):
        application_path = os.path.dirname(sys.executable)
    else:
        application_path = os.path.dirname(os.path.abspath(__file__))

    # Possible Tesseract locations (in priority order)
    possible_paths = [
        # 1. Bundled with executable
        os.path.join(application_path, 'tesseract', 'tesseract.exe'),

        # 2. In same directory as executable
        os.path.join(application_path, 'tesseract.exe'),

        # 3. Standard installation paths
        r'C:\Program Files\Tesseract-OCR\tesseract.exe',
        r'C:\Program Files (x86)\Tesseract-OCR\tesseract.exe',

        # 4. User AppData
        os.path.join(os.environ.get('LOCALAPPDATA', ''), 'Programs', 'Tesseract-OCR', 'tesseract.exe'),

        # 5. Environment PATH (Linux/Mac)
        'tesseract',
    ]

    for path in possible_paths:
        if os.path.exists(path):
            return path

    return None


def configure_tesseract():
    """Configure pytesseract with the correct path"""
    tesseract_path = find_tesseract()

    if tesseract_path:
        pytesseract.pytesseract.tesseract_cmd = tesseract_path
        return True

    return False


def check_tesseract_available():
    """Check if Tesseract is available and working"""
    try:
        version = pytesseract.get_tesseract_version()
        return True
    except:
        return False


def show_tesseract_error():
    """Show helpful error message if Tesseract is not found"""
    from tkinter import messagebox

    message = """Tesseract OCR is not installed or cannot be found.

The Detailed OCR Scanner requires Tesseract OCR to read text from the game.

Options:
1. Install Tesseract OCR from:
   https://github.com/UB-Mannheim/tesseract/wiki

   Recommended installer:
   tesseract-ocr-w64-setup-5.3.3.20231005.exe

2. Or use Fast Screenshot Scanner mode instead
   (does not require Tesseract)

After installing Tesseract, restart this application.
"""

    messagebox.showwarning("Tesseract OCR Not Found", message)


# Initialize on import
tesseract_configured = configure_tesseract()