import threading
import tkinter as tk
from tkinter import ttk, messagebox, filedialog
import json
import os
import sys

from SourceCode.FastScreenShotScanner import FastScreenshotScanner
from SourceCode.GameMapScanner import GameMapScanner
from SourceCode.tesseract_setup import check_tesseract_available, show_tesseract_error


class ScannerGUI():
    """Main application GUI"""

    def __init__(self, config):
        self.config = config

        # Determine config file location
        if getattr(sys, 'frozen', False):
            # Running as compiled executable
            self.application_path = os.path.dirname(sys.executable)
        else:
            # Running as script
            self.application_path = os.path.dirname(os.path.abspath(__file__))

        self.config_file = os.path.join(self.application_path, 'scanner_config.json')

        self.scanner = GameMapScanner(config)
        self.scanner.configure_standard_multi_tile_objects()
        self.fast_scanner = FastScreenshotScanner(config)

        self.root = tk.Tk()
        self.root.title("Game Map Scanner")
        self.root.geometry("850x1000")

        # Set window icon
        self._set_icon()

        self.scan_mode = tk.StringVar(value="detailed")

        self.root.bind('<Escape>', lambda e: self.stop_scan())  # Stops the scanning

        self.create_widgets()
        self.scan_thread = None

    def _set_icon(self):
        """Set the window icon"""
        icon_path = os.path.join(self.application_path, 'icon.ico')
        if os.path.exists(icon_path):
            try:
                self.root.iconbitmap(icon_path)
            except:
                pass  # Icon not found or invalid

    def create_widgets(self):
        # Menu bar
        menubar = tk.Menu(self.root)
        self.root.config(menu=menubar)

        # File menu
        file_menu = tk.Menu(menubar, tearoff=0)
        menubar.add_cascade(label="File", menu=file_menu)
        file_menu.add_command(label="Save Results", command=self.save_results)
        file_menu.add_command(label="Open Visualizer", command=self.open_visualizer)
        file_menu.add_separator()
        file_menu.add_command(label="Exit", command=self.root.quit)

        # Settings menu
        settings_menu = tk.Menu(menubar, tearoff=0)
        menubar.add_cascade(label="Settings", menu=settings_menu)
        settings_menu.add_command(label="Reconfigure Scanner", command=self.reconfigure)
        settings_menu.add_command(label="Edit Config File", command=self.edit_config)

        # Help menu
        help_menu = tk.Menu(menubar, tearoff=0)
        menubar.add_cascade(label="Help", menu=help_menu)
        help_menu.add_command(label="About", command=self.show_about)

        # Mode Selection
        mode_frame = ttk.LabelFrame(self.root, text="Scanner Mode", padding="10")
        mode_frame.pack(fill=tk.X, padx=10, pady=5)

        ttk.Radiobutton(mode_frame, text="Detailed OCR Scanner (slow, reads all data)",
                        variable=self.scan_mode, value="detailed").pack(anchor=tk.W)
        ttk.Radiobutton(mode_frame, text="Fast Screenshot Scanner (quick, stitches map images)",
                        variable=self.scan_mode, value="fast").pack(anchor=tk.W)

        # Control Frame
        control_frame = ttk.LabelFrame(self.root, text="Scan Controls", padding="10")
        control_frame.pack(fill=tk.X, padx=10, pady=5)

        # Coordinates
        ttk.Label(control_frame, text="Start X:").grid(row=0, column=0, padx=5)
        self.start_x = ttk.Entry(control_frame, width=10)
        self.start_x.insert(0, "0")
        self.start_x.grid(row=0, column=1, padx=5)

        ttk.Label(control_frame, text="Start Y:").grid(row=0, column=2, padx=5)
        self.start_y = ttk.Entry(control_frame, width=10)
        self.start_y.insert(0, "0")
        self.start_y.grid(row=0, column=3, padx=5)

        ttk.Label(control_frame, text="End X:").grid(row=1, column=0, padx=5)
        self.end_x = ttk.Entry(control_frame, width=10)
        self.end_x.insert(0, "100")
        self.end_x.grid(row=1, column=1, padx=5)

        ttk.Label(control_frame, text="End Y:").grid(row=1, column=2, padx=5)
        self.end_y = ttk.Entry(control_frame, width=10)
        self.end_y.insert(0, "100")
        self.end_y.grid(row=1, column=3, padx=5)

        ttk.Label(control_frame, text="Step (detailed mode):").grid(row=2, column=0, padx=5)
        self.step = ttk.Entry(control_frame, width=10)
        self.step.insert(0, "1")
        self.step.grid(row=2, column=1, padx=5)

        # Buttons
        self.start_btn = ttk.Button(control_frame, text="Start Scan", command=self.start_scan)
        self.start_btn.grid(row=3, column=0, columnspan=2, pady=10)

        self.stop_btn = ttk.Button(control_frame, text="Stop Scan", command=self.stop_scan, state=tk.DISABLED)
        self.stop_btn.grid(row=3, column=2, columnspan=2, pady=10)

        # Fast mode options
        fast_frame = ttk.LabelFrame(self.root, text="Fast Scanner Options", padding="10")
        fast_frame.pack(fill=tk.X, padx=10, pady=5)

        ttk.Button(fast_frame, text="Stitch Screenshots",
                   command=self.stitch_screenshots).pack(side=tk.LEFT, padx=5)
        ttk.Button(fast_frame, text="View Stitched Image",
                   command=self.view_stitched).pack(side=tk.LEFT, padx=5)

        # Progress
        progress_frame = ttk.LabelFrame(self.root, text="Progress", padding="10")
        progress_frame.pack(fill=tk.X, padx=10, pady=5)

        self.progress = ttk.Progressbar(progress_frame, length=750, mode='determinate')
        self.progress.pack(pady=5)

        self.status_label = ttk.Label(progress_frame, text="Ready")
        self.status_label.pack()

        # Results
        results_frame = ttk.LabelFrame(self.root, text="Results", padding="10")
        results_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=5)

        self.results_text = tk.Text(results_frame, height=12, width=95)
        self.results_text.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

        scrollbar = ttk.Scrollbar(results_frame, command=self.results_text.yview)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        self.results_text.config(yscrollcommand=scrollbar.set)

        # Action buttons
        button_frame = ttk.Frame(self.root)
        button_frame.pack(pady=10)

        ttk.Button(button_frame, text="Save Results", command=self.save_results).pack(side=tk.LEFT, padx=5)
        ttk.Button(button_frame, text="Open Visualizer", command=self.open_visualizer).pack(side=tk.LEFT, padx=5)

    def update_progress(self, percent, scanned, total_or_skipped):
        self.progress['value'] = percent
        if self.scan_mode.get() == "detailed":
            self.status_label.config(text=f"Scanned: {scanned} | Skipped: {total_or_skipped} | {percent:.1f}%")
        else:
            self.status_label.config(text=f"Captured: {scanned}/{total_or_skipped} screenshots | {percent:.1f}%")
        self.root.update_idletasks()

    def log_result(self, message):
        self.results_text.insert(tk.END, message + "\n")
        self.results_text.see(tk.END)
        self.root.update_idletasks()

    def start_scan(self):
        # Check if OCR mode and Tesseract is available
        if self.scan_mode.get() == "detailed" and not check_tesseract_available():
            show_tesseract_error()
            return

        try:
            start_x = int(self.start_x.get())
            start_y = int(self.start_y.get())
            end_x = int(self.end_x.get())
            end_y = int(self.end_y.get())
            step = int(self.step.get())
        except ValueError:
            messagebox.showerror("Error", "Invalid coordinates")
            return

        self.start_btn.config(state=tk.DISABLED)
        self.stop_btn.config(state=tk.NORMAL)
        self.results_text.delete(1.0, tk.END)

        # Check which mode is selected
        if self.scan_mode.get() == "detailed":
            def scan_thread():
                try:
                    self.log_result("Starting detailed OCR scan...")
                    self.scanner.scan_area(start_x, start_y, end_x, end_y, step, self.update_progress)
                    self.log_result("Scan complete!")
                except Exception as e:
                    self.log_result(f"Error during scan: {str(e)}")
                finally:
                    self.start_btn.config(state=tk.NORMAL)
                    self.stop_btn.config(state=tk.DISABLED)
        else:  # Fast mode
            def scan_thread():
                try:
                    self.log_result("Starting fast screenshot scan...")
                    self.log_result("Zooming out map...")
                    self.fast_scanner.scan_map_area(start_x, start_y, end_x, end_y, self.update_progress)
                    self.log_result("Screenshot capture complete!")
                    self.log_result(f"Captured {len(self.fast_scanner.screenshots)} screenshots")
                    self.log_result("Use 'Stitch Screenshots' to combine images")
                except Exception as e:
                    self.log_result(f"Error during scan: {str(e)}")
                finally:
                    self.start_btn.config(state=tk.NORMAL)
                    self.stop_btn.config(state=tk.DISABLED)

        self.scan_thread = threading.Thread(target=scan_thread, daemon=True)
        self.scan_thread.start()

    def stop_scan(self):
        if self.scan_mode.get() == "detailed":
            self.scanner.stop()
        else:
            self.fast_scanner.stop()

        self.stop_btn.config(state=tk.DISABLED)
        self.start_btn.config(state=tk.NORMAL)
        self.log_result("Scan stopped by user")

    def save_results(self):
        filename = filedialog.asksaveasfilename(
            defaultextension=".json",
            filetypes=[("JSON files", "*.json")]
        )
        if filename:
            self.scanner.save_data(filename)
            messagebox.showinfo("Success", f"Results saved to {filename}")

    def stitch_screenshots(self):
        """Stitch fast scanner screenshots"""
        if not self.fast_scanner.screenshots:
            messagebox.showwarning("No Data", "No screenshots to stitch. Run a fast scan first.")
            return

        filename = filedialog.asksaveasfilename(
            defaultextension=".png",
            filetypes=[("PNG files", "*.png")]
        )
        if filename:
            self.fast_scanner.stitch_screenshots(filename)
            messagebox.showinfo("Success", f"Map stitched to {filename}")

    def view_stitched(self):
        """View stitched image"""
        filename = filedialog.askopenfilename(
            filetypes=[("Image files", "*.png *.jpg")]
        )
        if filename:
            import os
            os.startfile(filename)  # Opens with default image viewer

    def open_visualizer(self):
        """Open the map visualizer in a new window"""
        try:
            from SourceCode.MapVisualizer import MapVisualizer
            visualizer = MapVisualizer()

            # Run in separate thread to not block main window
            import threading
            viz_thread = threading.Thread(target=visualizer.run, daemon=True)
            viz_thread.start()
        except Exception as e:
            messagebox.showerror("Error", f"Failed to open visualizer:\n{str(e)}")

    def reconfigure(self):
        """Rerun the configuration wizard"""
        response = messagebox.askyesno(
            "Reconfigure",
            "This will close the current application and rerun the configuration wizard.\n\n"
            "Your current configuration will be backed up.\n\nContinue?"
        )

        if response:
            # Backup current config
            try:
                import shutil
                if os.path.exists(self.config_file):
                    backup_file = os.path.join(self.application_path, 'scanner_config_backup.json')
                    shutil.copy(self.config_file, backup_file)
                    messagebox.showinfo("Backup", f"Current config backed up to {backup_file}")
            except Exception as e:
                messagebox.showwarning("Backup Failed", f"Could not backup config: {e}")

            # Close current window
            self.root.destroy()

            # Run wizard
            from SourceCode.ConfigWizard import ConfigWizard
            wizard = ConfigWizard()
            new_config = wizard.run()

            if new_config:
                # Restart with new config
                new_app = ScannerGUI(new_config)
                new_app.run()
            else:
                messagebox.showinfo("Cancelled", "Reconfiguration cancelled. Application will exit.")

    def edit_config(self):
        """Open config file in default text editor"""
        if os.path.exists(self.config_file):
            try:
                os.startfile(self.config_file)
            except:
                messagebox.showinfo("Config Location", f"Config file location:\n{os.path.abspath(self.config_file)}")
        else:
            messagebox.showwarning("Not Found", "Configuration file not found.")

    def show_about(self):
        """Show about dialog"""
        about_text = """Game Map Scanner v1.0

A tool for scanning and mapping game worlds.

Features:
• Detailed OCR scanning
• Fast screenshot scanning
• Map visualization
• Data export

Press ESC during scanning to stop.
        """
        messagebox.showinfo("About", about_text)

    def run(self):
        self.root.mainloop()