import pyautogui
import tkinter as tk
from tkinter import ttk, messagebox
from pynput import keyboard


class PositionCapture():
    """Helper class to capture screen positions as percentages"""

    @staticmethod
    def capture_position(label_text, parent=None):
        """Capture a screen position and return as percentage of screen size"""
        screen_w = pyautogui.size()[0]
        screen_h = pyautogui.size()[1]

        msg = f"{label_text}\n\nMove your mouse to the position and press SPACE\nPress ESC to cancel"

        if parent:
            capture_window = tk.Toplevel(parent)
        else:
            capture_window = tk.Tk()

        capture_window.title("Position Capture")
        capture_window.geometry("600x200")
        capture_window.attributes('-topmost', True)

        # Make window stay on top and grab focus
        capture_window.lift()
        capture_window.focus_force()
        capture_window.attributes('-topmost', True)

        label = tk.Label(capture_window, text=msg, pady=20, font=('Arial', 10))
        label.pack()

        coord_label = tk.Label(capture_window, text="", font=('Arial', 12, 'bold'))
        coord_label.pack()

        result = {'pos': None, 'cancelled': False, 'running': True, 'listener': None}

        def update_coords():
            if result['running'] and result['pos'] is None and not result['cancelled']:
                try:
                    x, y = pyautogui.position()
                    coord_label.config(text=f"Current: ({x}, {y})")
                    capture_window.after(50, update_coords)
                except tk.TclError:
                    result['running'] = False
                except Exception:
                    pass

        def on_key_press(key):
            """Handle keyboard events using pynput (global hotkeys)"""
            try:
                if key == keyboard.Key.space and result['running']:
                    result['running'] = False
                    x, y = pyautogui.position()
                    result['pos'] = (x / screen_w, y / screen_h)
                    try:
                        capture_window.quit()
                    except:
                        pass
                elif key == keyboard.Key.esc:
                    result['running'] = False
                    result['cancelled'] = True
                    try:
                        capture_window.quit()
                    except:
                        pass
            except AttributeError:
                pass

        def on_close():
            result['running'] = False
            result['cancelled'] = True
            if result['listener']:
                result['listener'].stop()
            try:
                capture_window.quit()
            except:
                pass

        capture_window.protocol("WM_DELETE_WINDOW", on_close)

        # Keep window focused
        def keep_focus():
            if result['running']:
                try:
                    capture_window.lift()
                    capture_window.attributes('-topmost', True)
                    capture_window.after(500, keep_focus)
                except:
                    pass

        # Start global keyboard listener
        try:
            result['listener'] = keyboard.Listener(on_press=on_key_press)
            result['listener'].start()
        except Exception as e:
            messagebox.showerror("Error", f"Could not start keyboard listener:\n{str(e)}")
            capture_window.destroy()
            return None

        keep_focus()
        update_coords()
        capture_window.mainloop()

        # Stop listener
        if result['listener']:
            result['listener'].stop()

        try:
            capture_window.destroy()
        except:
            pass

        return result['pos'] if not result['cancelled'] else None

    @staticmethod
    def capture_region(label_text, parent=None):
        """Capture a screen region and return as percentage"""
        screen_w = pyautogui.size()[0]
        screen_h = pyautogui.size()[1]

        msg = f"{label_text}\n\nClick top-left corner, then bottom-right corner\nPress ESC to cancel"

        if parent:
            capture_window = tk.Toplevel(parent)
        else:
            capture_window = tk.Tk()

        capture_window.title("Region Capture")
        capture_window.geometry("600x200")
        capture_window.attributes('-topmost', True)
        capture_window.lift()
        capture_window.focus_force()

        label = tk.Label(capture_window, text=msg, pady=20, font=('Arial', 10))
        label.pack()

        coord_label = tk.Label(capture_window, text="Waiting for first corner...",
                               font=('Arial', 12, 'bold'))
        coord_label.pack()

        result = {'corners': [], 'cancelled': False, 'mouse_listener': None,
                  'key_listener': None, 'running': True}

        def on_click(x, y, button, pressed):
            if pressed and not result['cancelled'] and result['running']:
                result['corners'].append((x / screen_w, y / screen_h))
                if len(result['corners']) == 1:
                    try:
                        coord_label.config(text=f"First corner: ({x}, {y})\nClick second corner...")
                    except:
                        pass
                elif len(result['corners']) == 2:
                    result['running'] = False
                    if result['mouse_listener']:
                        result['mouse_listener'].stop()
                    if result['key_listener']:
                        result['key_listener'].stop()
                    try:
                        capture_window.quit()
                    except:
                        pass
                    return False

        def on_key_press(key):
            try:
                if key == keyboard.Key.esc:
                    result['running'] = False
                    result['cancelled'] = True
                    if result['mouse_listener']:
                        result['mouse_listener'].stop()
                    if result['key_listener']:
                        result['key_listener'].stop()
                    try:
                        capture_window.quit()
                    except:
                        pass
            except AttributeError:
                pass

        def on_close():
            result['running'] = False
            result['cancelled'] = True
            if result['mouse_listener']:
                result['mouse_listener'].stop()
            if result['key_listener']:
                result['key_listener'].stop()
            try:
                capture_window.quit()
            except:
                pass

        capture_window.protocol("WM_DELETE_WINDOW", on_close)

        # Keep window on top
        def keep_focus():
            if result['running']:
                try:
                    capture_window.lift()
                    capture_window.attributes('-topmost', True)
                    capture_window.after(500, keep_focus)
                except:
                    pass

        try:
            from pynput import mouse
            result['mouse_listener'] = mouse.Listener(on_click=on_click)
            result['mouse_listener'].start()

            result['key_listener'] = keyboard.Listener(on_press=on_key_press)
            result['key_listener'].start()
        except ImportError:
            messagebox.showerror("Error", "pynput library not installed.\nRun: pip install pynput")
            try:
                capture_window.destroy()
            except:
                pass
            return None

        keep_focus()
        capture_window.mainloop()

        if result['mouse_listener']:
            result['mouse_listener'].stop()
        if result['key_listener']:
            result['key_listener'].stop()

        try:
            capture_window.destroy()
        except:
            pass

        if result['cancelled'] or len(result['corners']) != 2:
            return None

        x1, y1 = result['corners'][0]
        x2, y2 = result['corners'][1]

        return (min(x1, x2), min(y1, y2), abs(x2 - x1), abs(y2 - y1))

    @staticmethod
    def capture_zoom_control(parent=None):
        """Capture zoom slider position"""
        msg = "Capture zoom slider (horizontal position for zoom level)"
        return PositionCapture.capture_position(msg, parent)