import json
import tkinter as tk
from tkinter import ttk, filedialog, messagebox
from PIL import Image, ImageDraw, ImageTk, ImageFont
import math
import os
import sys


class MapVisualizer:
    """Interactive map visualizer for scanned game map data"""

    def __init__(self):
        self.root = tk.Tk()
        self.root.title("Game Map Visualizer")
        self.root.geometry("1400x900")

        self.map_data = {}
        self.canvas_img = None
        self.photo_img = None
        self.canvas_image_id = None
        self.tile_size = 5  # Pixels per tile
        self.banner_range = 5  # Default banner range in tiles (square area)

        # Visibility toggles
        self.show_layers = {
            'land': tk.BooleanVar(value=True),
            'players': tk.BooleanVar(value=True),
            'wildlife': tk.BooleanVar(value=True),
            'resources': tk.BooleanVar(value=True),
            'alliance_buildings': tk.BooleanVar(value=True),
            'player_buildings': tk.BooleanVar(value=True),
            'special_buildings': tk.BooleanVar(value=True),
            'banner_radius': tk.BooleanVar(value=False),
            'grid': tk.BooleanVar(value=False),
            'labels': tk.BooleanVar(value=True),
        }

        # Color scheme
        self.colors = {
            'land': '#8B7355',
            'plains': '#90EE90',
            'badland': '#8B4513',
            'ruins': '#696969',
            'fertile_land': '#32CD32',
            'forbidden_area': '#4B0082',

            'playerGood': '#00FF00',
            'playerBad': '#FF0000',

            'wildlifeSmall': '#FFD700',
            'wildlifeBig': '#FF8C00',

            'idle_resource': '#87CEEB',
            'alliance_resource': '#4169E1',
            'alliance_banner': '#9370DB',

            'hunting_trap': '#8B4513',
            'hq': '#FF1493',
            'arsenal': '#DC143C',
            'harvest altar': '#FFD700',
            'frontier lodge': '#8B4513',
            "scholar's tower": '#4169E1',
            'drill camp': '#696969',
            'forager grove': '#228B22',

            'special_building': '#800080',
            "king's castle": '#FFD700',
            'fortress': '#8B0000',
            'sanctuary': '#FFFFFF',
            'turret': '#708090',

            'buff_location': '#00FFFF',

            'banner_radius': '#9370DB',
            'grid': '#CCCCCC',
        }

        self.create_widgets()

    def _set_icon(self):
        """Set the window icon"""
        try:
            import sys
            if getattr(sys, 'frozen', False):
                application_path = os.path.dirname(sys.executable)
            else:
                application_path = os.path.dirname(os.path.abspath(__file__))

            icon_path = os.path.join(application_path, 'icon.ico')
            if os.path.exists(icon_path):
                self.root.iconbitmap(icon_path)
        except:
            pass  # Icon not found or invalid

    def create_widgets(self):
        # Top control frame
        control_frame = ttk.Frame(self.root)
        control_frame.pack(side=tk.TOP, fill=tk.X, padx=10, pady=5)

        ttk.Button(control_frame, text="Load Map Data (JSON)",
                   command=self.load_data).pack(side=tk.LEFT, padx=5)
        ttk.Button(control_frame, text="Export Image",
                   command=self.export_image).pack(side=tk.LEFT, padx=5)
        ttk.Button(control_frame, text="Visualize Map",
                   command=self.open_visualizer).pack(side=tk.LEFT, padx=5)

        ttk.Label(control_frame, text="Tile Size:").pack(side=tk.LEFT, padx=5)
        self.tile_size_var = tk.IntVar(value=5)
        tile_size_spin = ttk.Spinbox(control_frame, from_=1, to=20,
                                     textvariable=self.tile_size_var, width=5)
        tile_size_spin.pack(side=tk.LEFT, padx=5)

        ttk.Label(control_frame, text="Banner Range:").pack(side=tk.LEFT, padx=5)
        self.banner_range_var = tk.IntVar(value=5)
        banner_spin = ttk.Spinbox(control_frame, from_=1, to=20,
                                  textvariable=self.banner_range_var, width=5)
        banner_spin.pack(side=tk.LEFT, padx=5)

        ttk.Button(control_frame, text="Redraw Map",
                   command=self.draw_map).pack(side=tk.LEFT, padx=5)

        # Main content frame
        main_frame = ttk.Frame(self.root)
        main_frame.pack(side=tk.TOP, fill=tk.BOTH, expand=True, padx=10, pady=5)

        # Legend frame (left side)
        legend_frame = ttk.LabelFrame(main_frame, text="Legend & Controls", padding="10")
        legend_frame.pack(side=tk.LEFT, fill=tk.Y, padx=(0, 5))

        # Create scrollable legend
        legend_canvas = tk.Canvas(legend_frame, width=250)
        legend_scrollbar = ttk.Scrollbar(legend_frame, orient="vertical",
                                         command=legend_canvas.yview)
        legend_content = ttk.Frame(legend_canvas)

        legend_content.bind(
            "<Configure>",
            lambda e: legend_canvas.configure(scrollregion=legend_canvas.bbox("all"))
        )

        legend_canvas.create_window((0, 0), window=legend_content, anchor="nw")
        legend_canvas.configure(yscrollcommand=legend_scrollbar.set)

        legend_canvas.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        legend_scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        # Add legend items
        self.create_legend_items(legend_content)

        # Map canvas frame (right side)
        canvas_frame = ttk.Frame(main_frame)
        canvas_frame.pack(side=tk.RIGHT, fill=tk.BOTH, expand=True)

        # Canvas with scrollbars
        self.canvas = tk.Canvas(canvas_frame, bg='white')
        h_scrollbar = ttk.Scrollbar(canvas_frame, orient=tk.HORIZONTAL,
                                    command=self.canvas.xview)
        v_scrollbar = ttk.Scrollbar(canvas_frame, orient=tk.VERTICAL,
                                    command=self.canvas.yview)

        self.canvas.configure(xscrollcommand=h_scrollbar.set,
                              yscrollcommand=v_scrollbar.set)

        self.canvas.grid(row=0, column=0, sticky="nsew")
        h_scrollbar.grid(row=1, column=0, sticky="ew")
        v_scrollbar.grid(row=0, column=1, sticky="ns")

        canvas_frame.grid_rowconfigure(0, weight=1)
        canvas_frame.grid_columnconfigure(0, weight=1)

        # Status bar
        self.status_label = ttk.Label(self.root, text="Load a map data file to begin",
                                      relief=tk.SUNKEN)
        self.status_label.pack(side=tk.BOTTOM, fill=tk.X)

    def create_legend_items(self, parent):
        """Create legend checkboxes with color indicators"""

        categories = [
            ("Terrain", [
                ('land', 'Land/Terrain'),
                ('grid', 'Grid Lines'),
            ]),
            ("Players", [
                ('players', 'Player Cities'),
            ]),
            ("Wildlife", [
                ('wildlife', 'Wildlife'),
            ]),
            ("Resources", [
                ('resources', 'Resource Nodes'),
            ]),
            ("Alliance", [
                ('alliance_buildings', 'Alliance Buildings'),
                ('banner_radius', 'Banner Area of Effect'),
            ]),
            ("Buildings", [
                ('player_buildings', 'Player Buildings'),
                ('special_buildings', 'Special Buildings'),
            ]),
            ("Other", [
                ('labels', 'Show Labels'),
            ]),
        ]

        for category_name, items in categories:
            ttk.Label(parent, text=category_name, font=('Arial', 10, 'bold')).pack(
                anchor=tk.W, pady=(10, 5))

            for key, label in items:
                frame = ttk.Frame(parent)
                frame.pack(anchor=tk.W, pady=2)

                cb = ttk.Checkbutton(frame, text=label, variable=self.show_layers[key],
                                     command=self.draw_map)
                cb.pack(side=tk.LEFT)

        # Color reference
        ttk.Label(parent, text="Color Reference",
                  font=('Arial', 10, 'bold')).pack(anchor=tk.W, pady=(20, 5))

        color_items = [
            ('Plains', 'plains'),
            ('Badland', 'badland'),
            ('Ruins', 'ruins'),
            ('Friendly Player', 'playerGood'),
            ('Enemy Player', 'playerBad'),
            ('Small Wildlife', 'wildlifeSmall'),
            ('Large Wildlife', 'wildlifeBig'),
            ('Alliance Banner', 'alliance_banner'),
            ('Special Building', 'special_building'),
        ]

        for name, color_key in color_items:
            frame = ttk.Frame(parent)
            frame.pack(anchor=tk.W, pady=2)

            color_box = tk.Canvas(frame, width=20, height=15, bg=self.colors[color_key],
                                  highlightthickness=1, highlightbackground='black')
            color_box.pack(side=tk.LEFT, padx=(0, 5))

            ttk.Label(frame, text=name).pack(side=tk.LEFT)

    def load_data(self):
        """Load map data from JSON file"""
        filename = filedialog.askopenfilename(
            title="Select Map Data File",
            filetypes=[("JSON files", "*.json"), ("All files", "*.*")]
        )

        if not filename:
            return

        try:
            with open(filename, 'r') as f:
                self.map_data = json.load(f)

            if not self.map_data:
                messagebox.showwarning("Empty File", "The JSON file contains no map data")
                return

            self.status_label.config(text=f"Loaded {len(self.map_data)} locations from {filename}")
            self.draw_map()
        except json.JSONDecodeError as e:
            messagebox.showerror("Error", f"Invalid JSON file:\n{str(e)}")
        except Exception as e:
            messagebox.showerror("Error", f"Failed to load map data:\n{str(e)}")

    def draw_map(self):
        """Draw the complete map"""
        if not self.map_data:
            messagebox.showwarning("No Data", "Please load map data first")
            return

        try:
            self.tile_size = self.tile_size_var.get()
            self.banner_range = self.banner_range_var.get()

            # Find map bounds
            coords = []
            for coord_str in self.map_data.keys():
                try:
                    x, y = map(int, coord_str.split(','))
                    coords.append((x, y))
                except (ValueError, AttributeError) as e:
                    print(f"Skipping invalid coordinate: {coord_str}")
                    continue

            if not coords:
                messagebox.showerror("Error", "No valid coordinates found in map data")
                return

            min_x = min(c[0] for c in coords)
            max_x = max(c[0] for c in coords)
            min_y = min(c[1] for c in coords)
            max_y = max(c[1] for c in coords)

            width = (max_x - min_x + 1) * self.tile_size
            height = (max_y - min_y + 1) * self.tile_size

            # Validate dimensions
            if width <= 0 or height <= 0:
                messagebox.showerror("Error", "Invalid map dimensions")
                return

            if width > 10000 or height > 10000:
                response = messagebox.askyesno(
                    "Large Map",
                    f"Map is very large ({width}x{height}px). This may take a while. Continue?"
                )
                if not response:
                    return

            # Create image
            img = Image.new('RGB', (width, height), 'white')
            draw = ImageDraw.Draw(img, 'RGBA')

            # Try to load a font
            try:
                font = ImageFont.truetype("arial.ttf", max(8, self.tile_size))
                small_font = ImageFont.truetype("arial.ttf", max(6, self.tile_size - 2))
            except:
                font = ImageFont.load_default()
                small_font = font

            # Draw banner areas first (so they're behind everything)
            if self.show_layers['banner_radius'].get():
                self.draw_banner_areas(draw, coords, min_x, min_y, max_y)

            # Draw grid
            if self.show_layers['grid'].get():
                self.draw_grid(draw, width, height, min_x, min_y, max_x, max_y)

            # Draw tiles
            for coord_str, location in self.map_data.items():
                try:
                    x, y = map(int, coord_str.split(','))
                    px = (x - min_x) * self.tile_size
                    py = (max_y - y) * self.tile_size  # Flip Y axis

                    color = self.get_location_color(location)
                    if color:
                        draw.rectangle(
                            [px, py, px + self.tile_size - 1, py + self.tile_size - 1],
                            fill=color,
                            outline='black' if self.tile_size > 3 else None
                        )

                    # Draw labels for important items
                    if self.show_layers['labels'].get() and self.tile_size >= 5:
                        self.draw_label(draw, location, px, py, small_font)
                except (ValueError, AttributeError, KeyError) as e:
                    print(f"Error drawing tile {coord_str}: {e}")
                    continue

            # Convert to PhotoImage and display
            self.canvas_img = img

            # Delete old image from canvas if it exists
            if self.canvas_image_id:
                self.canvas.delete(self.canvas_image_id)
            else:
                self.canvas.delete("all")

            # Create PhotoImage - must keep reference!
            self.photo_img = ImageTk.PhotoImage(img)

            # Create the image on canvas and store its ID
            self.canvas_image_id = self.canvas.create_image(0, 0, anchor=tk.NW, image=self.photo_img)
            self.canvas.configure(scrollregion=(0, 0, width, height))

            self.status_label.config(
                text=f"Map: {max_x - min_x + 1}x{max_y - min_y + 1} tiles | "
                     f"{len(self.map_data)} locations | "
                     f"Image: {width}x{height}px"
            )

        except Exception as e:
            messagebox.showerror("Error", f"Failed to draw map:\n{str(e)}")
            import traceback
            traceback.print_exc()

    def draw_grid(self, draw, width, height, min_x, min_y, max_x, max_y):
        """Draw grid lines every 10 tiles"""
        grid_interval = 10 * self.tile_size

        # Vertical lines
        for x in range(0, width, grid_interval):
            draw.line([(x, 0), (x, height)], fill=self.colors['grid'], width=1)

        # Horizontal lines
        for y in range(0, height, grid_interval):
            draw.line([(0, y), (width, y)], fill=self.colors['grid'], width=1)

    def draw_banner_areas(self, draw, coords, min_x, min_y, max_y):
        """Draw square areas around alliance banners"""
        for coord_str, location in self.map_data.items():
            if location.get('type') == 'alliance_banner':
                x, y = map(int, coord_str.split(','))

                # Calculate the square area (banner_range tiles in each direction)
                x1 = x - self.banner_range
                y1 = y - self.banner_range
                x2 = x + self.banner_range
                y2 = y + self.banner_range

                # Convert to pixel coordinates
                px1 = (x1 - min_x) * self.tile_size
                py1 = (max_y - y2) * self.tile_size  # Flip Y
                px2 = (x2 - min_x + 1) * self.tile_size
                py2 = (max_y - y1 + 1) * self.tile_size  # Flip Y

                # Draw semi-transparent square overlay
                draw.rectangle(
                    [px1, py1, px2, py2],
                    fill=(*self._hex_to_rgb(self.colors['banner_radius']), 50),  # Semi-transparent
                    outline=self.colors['banner_radius'],
                    width=2
                )

    def draw_label(self, draw, location, px, py, font):
        """Draw label for special locations"""
        label_types = {'alliance_banner', 'special_building', 'hq',
                       "king's castle", 'fortress', 'sanctuary'}

        if location.get('type') in label_types and self.tile_size >= 8:
            name = location.get('name', '')
            if name:
                # Abbreviate long names
                if len(name) > 15:
                    name = name[:12] + "..."

                text_bbox = draw.textbbox((0, 0), name, font=font)
                text_width = text_bbox[2] - text_bbox[0]
                text_height = text_bbox[3] - text_bbox[1]

                text_x = px + self.tile_size // 2 - text_width // 2
                text_y = py + self.tile_size // 2 - text_height // 2

                # Background
                draw.rectangle(
                    [text_x - 2, text_y - 1, text_x + text_width + 2, text_y + text_height + 1],
                    fill=(255, 255, 255, 200)
                )

                draw.text((text_x, text_y), name, fill='black', font=font)

    def _hex_to_rgb(self, hex_color):
        """Convert hex color to RGB tuple"""
        hex_color = hex_color.lstrip('#')
        return tuple(int(hex_color[i:i + 2], 16) for i in (0, 2, 4))

    def get_location_color(self, location):
        """Get color for a location based on type and visibility settings"""
        loc_type = location.get('type', 'unknown')

        # Check visibility
        if loc_type == 'land':
            if not self.show_layers['land'].get():
                return None
            # Get specific land type
            name = location.get('name', '').lower()
            if 'plains' in name:
                return self.colors['plains']
            elif 'badland' in name:
                return self.colors['badland']
            elif 'ruins' in name:
                return self.colors['ruins']
            elif 'fertile' in name:
                return self.colors['fertile_land']
            elif 'forbidden' in name:
                return self.colors['forbidden_area']
            return self.colors['land']

        elif loc_type in ['playerGood', 'playerBad']:
            if not self.show_layers['players'].get():
                return None
            return self.colors[loc_type]

        elif loc_type in ['wildlifeSmall', 'wildlifeBig']:
            if not self.show_layers['wildlife'].get():
                return None
            return self.colors[loc_type]

        elif loc_type in ['idle_resource']:
            if not self.show_layers['resources'].get():
                return None
            return self.colors['idle_resource']

        elif loc_type in ['alliance_resource', 'alliance_banner']:
            if not self.show_layers['alliance_buildings'].get():
                return None
            return self.colors.get(loc_type, self.colors['alliance_resource'])

        elif loc_type in ['hunting_trap', 'hq', 'arsenal', 'harvest altar',
                          'frontier lodge', "scholar's tower", 'drill camp',
                          'forager grove']:
            if not self.show_layers['player_buildings'].get():
                return None
            return self.colors.get(loc_type, '#FF1493')

        elif loc_type in ['special_building', "king's castle", 'fortress',
                          'sanctuary', 'turret']:
            if not self.show_layers['special_buildings'].get():
                return None
            return self.colors.get(loc_type, self.colors['special_building'])

        elif loc_type == 'buff_location':
            if not self.show_layers['alliance_buildings'].get():
                return None
            return self.colors['buff_location']

        return None

    def export_image(self):
        """Export current map as PNG"""
        if not self.canvas_img:
            messagebox.showwarning("No Map", "Please draw a map first")
            return

        filename = filedialog.asksaveasfilename(
            defaultextension=".png",
            filetypes=[("PNG files", "*.png"), ("JPEG files", "*.jpg")]
        )

        if filename:
            self.canvas_img.save(filename)
            messagebox.showinfo("Success", f"Map exported to {filename}")

    def run(self):
        """Run the visualizer"""
        self.root.mainloop()


def main():
    visualizer = MapVisualizer()
    visualizer.run()


if __name__ == "__main__":
    main()