import pyautogui
import pytesseract
import time
from PIL import Image, ImageEnhance
import json

from SourceCode.BaseClass import BaseClass

# Configure PyAutoGUI
pyautogui.PAUSE = 0.3
pyautogui.FAILSAFE = True

# Try to find Tesseract
try:
    pytesseract.get_tesseract_version()
except:
    pytesseract.pytesseract.tesseract_cmd = r'C:\Program Files\Tesseract-OCR\tesseract.exe'


class GameMapScanner(BaseClass):
    """Main scanner class with percentage-based positioning"""

    def __init__(self, config):
        super().__init__(config)

    def configure_standard_multi_tile_objects(self):
        """Configure standard multi-tile patterns"""
        pattern_2x2 = [(0, 1), (1, 0), (1, 1)]

        pattern_3x3 = [
                    (0, 1), (0, 2),
            (1, 0), (1, 1), (1, 2),
            (2, 0), (2, 1), (2, 2)
        ]
        pattern_6x6 = [
                    (0, 1), (0, 2), (0, 3), (0, 4), (0, 5),
            (1, 0), (1, 1), (1, 2), (1, 3), (1, 4), (1, 5),
            (2, 0), (2, 1), (2, 2), (2, 3), (2, 4), (2, 5),
            (3, 0), (3, 1), (3, 2), (3, 3), (3, 4), (3, 5),
            (4, 0), (4, 1), (4, 2), (4, 3), (4, 4), (4, 5),
            (5, 0), (5, 1), (5, 2), (5, 3), (5, 4), (5, 5),
        ]

        for obj_type in ['playerGood', 'playerBad', 'alliance_resource',
                         'wildlifeBig', 'buff_location']:
            self.multi_tile_objects[obj_type] = pattern_2x2

        for obj_type in ['hq', 'arsenal', 'harvest altar', 'frontier lodge',
                         "scholar's tower", 'drill camp', 'hunting_trap', 'forager grove']:
            self.multi_tile_objects[obj_type] = pattern_3x3

        for obj_type in ['special_building']:
            self.multi_tile_objects[obj_type] = pattern_6x6

    def add_adjacent_skips(self, x, y, object_type):
        """Add adjacent tiles to skip list"""
        if object_type not in self.multi_tile_objects:
            return 0

        pattern = self.multi_tile_objects[object_type]
        added = 0

        for x_offset, y_offset in pattern:
            skip_x = x + x_offset
            skip_y = y + y_offset
            if 0 <= skip_x <= 1199 and 0 <= skip_y <= 1199:
                self.skip_cords.add((skip_x, skip_y))
                added += 1

        return added

    def should_skip_coordinate(self, x, y):
        """Check if coordinate should be skipped"""
        if (x, y) in self.skip_cords:
            return True

        for region in self.skip_regions:
            if (region['start_x'] <= x <= region['end_x'] and
                region['start_y'] <= y <= region['end_y']):
                return True

        return False

    def click_center_and_read_popup(self):
        """Click center and read popup with enhanced OCR"""
        center = self._percent_to_pixel(*self.config['center_click'])
        pyautogui.click(*center)
        time.sleep(0.4)

        region = self._percent_region_to_pixel(self.config['popup_region'])
        screenshot = pyautogui.screenshot(region=region)

        # Enhance image for better OCR
        screenshot = ImageEnhance.Contrast(screenshot).enhance(1.5)
        screenshot = ImageEnhance.Sharpness(screenshot).enhance(1.5)

        text = pytesseract.image_to_string(screenshot)
        return text.strip()

    def close_popup(self):
        """Close popup"""
        close = self._percent_to_pixel(*self.config['close_popup'])
        pyautogui.click(*close)
        time.sleep(0.2)

    def parse_popup_text(self, text):
        """Parse popup text to identify location type"""
        text_lower = text.lower()
        result = {
            'type': 'unknown',
            'name': None,
            'level': None,
            'occupied_by': None,
            'gatherer': None,
            'capacity': None,
            'production_rate': None,
            'power': None,
            'alliance': None,
            'details': text
        }

        # Hunting Traps
        if 'hunting trap' in text_lower:
            result['type'] = 'hunting_trap'
            result['name'] = 'Hunting Trap'

        # Idle Resources
        elif 'idle iron mine' in text_lower:
            result['type'] = 'idle_resource'
            result['name'] = 'Idle Iron Mine'
        elif 'idle lumberyard' in text_lower:
            result['type'] = 'idle_resource'
            result['name'] = 'Idle Lumberyard'
        elif 'idle mill' in text_lower:
            result['type'] = 'idle_resource'
            result['name'] = 'Idle Mill'
        elif 'idle quarry' in text_lower:
            result['type'] = 'idle_resource'
            result['name'] = 'Idle Quarry'

        # Alliance Buildings
        elif 'alliance banner' in text_lower:
            result['type'] = 'alliance_banner'
            result['name'] = 'Alliance Banner'
        elif 'alliance iron mine' in text_lower:
            result['type'] = 'alliance_resource'
            result['name'] = 'Alliance Iron Mine'
        elif 'alliance mill' in text_lower:
            result['type'] = 'alliance_resource'
            result['name'] = 'Alliance Mill'
        elif 'alliance quarry' in text_lower:
            result['type'] = 'alliance_resource'
            result['name'] = 'Alliance Quarry'
        elif 'alliance wood mill' in text_lower:
            result['type'] = 'alliance_resource'
            result['name'] = 'Alliance Woodmill'

        # Player Buildings
        elif 'arsenal' in text_lower:
            result['type'] = 'arsenal'
            result['name'] = 'Arsenal'
        elif 'harvest altar' in text_lower:
            result['type'] = 'harvest altar'
            result['name'] = 'Harvest Altar'
        elif 'frontier lodge' in text_lower:
            result['type'] = 'frontier lodge'
            result['name'] = 'Frontier Lodge'
        elif "scholar's tower" in text_lower or 'scholars tower' in text_lower:
            result['type'] = "scholar's tower"
            result['name'] = "Scholar's Tower"
        elif 'drill camp' in text_lower:
            result['type'] = 'drill camp'
            result['name'] = 'Drill Camp'
        elif 'forager grove' in text_lower:
            result['type'] = 'forager grove'
            result['name'] = 'Forager Grove'

        # HQ
        elif 'badland hq' in text_lower:
            result['type'] = 'hq'
            result['name'] = 'Badland HQ'
        elif 'plains hq' in text_lower:
            result['type'] = 'hq'
            result['name'] = 'Plains HQ'

        # Special Buildings
        elif "king's castle" in text_lower or 'kings castle' in text_lower:
            result['type'] = "king's castle"
            result['name'] = "King's Castle"
        elif 'fortress' in text_lower:
            result['type'] = 'special_building'
            result['name'] = 'Fortress'
        elif 'sanctuary' in text_lower:
            result['type'] = 'special_building'
            result['name'] = 'Sanctuary'
        elif 'turret' in text_lower:
            result['type'] = 'turret'
            result['name'] = 'Turret'

        # Buff locations
        elif 'guardian' in text_lower:
            result['type'] = 'buff_location'
            result['name'] = 'Guardian'
        elif 'silver sentinel' in text_lower:
            result['type'] = 'buff_location'
            result['name'] = 'Silver Sentinel'

        # Land
        elif 'ruins' in text_lower:
            result['type'] = 'land'
            result['name'] = 'Ruins'
        elif 'badland' in text_lower and 'hq' not in text_lower:
            result['type'] = 'land'
            result['name'] = 'Badland'
        elif 'fertile land' in text_lower:
            result['type'] = 'land'
            result['name'] = 'Fertile Land'
        elif 'forbidden area' in text_lower:
            result['type'] = 'land'
            result['name'] = 'Forbidden Area'
        elif 'plains' in text_lower:
            result['type'] = 'land'
            result['name'] = 'Plains'

        # Wildlife
        elif 'ironclad war bear' in text_lower:
            result['type'] = 'wildlifeBig'
            result['name'] = 'Ironclad War Bear'
        elif 'ironclad war elephant' in text_lower:
            result['type'] = 'wildlifeBig'
            result['name'] = 'Ironclad War Elephant'
        elif 'regal white lion' in text_lower:
            result['type'] = 'wildlifeBig'
            result['name'] = 'Regal White Lion'
        elif 'alpha black panther' in text_lower:
            result['type'] = 'wildlifeBig'
            result['name'] = 'Alpha Black Panther'
        elif 'great moose' in text_lower:
            result['type'] = 'wildlifeBig'
            result['name'] = 'Great Moose'
        elif 'grizzly bear' in text_lower:
            result['type'] = 'wildlifeSmall'
            result['name'] = 'Grizzly Bear'
        elif 'cheetah' in text_lower:
            result['type'] = 'wildlifeSmall'
            result['name'] = 'Cheetah'
        elif 'gray wolf' in text_lower or 'grey wolf' in text_lower:
            result['type'] = 'wildlifeSmall'
            result['name'] = 'Gray Wolf'
        elif 'bison' in text_lower:
            result['type'] = 'wildlifeSmall'
            result['name'] = 'Bison'
        elif 'lion' in text_lower:
            result['type'] = 'wildlifeSmall'
            result['name'] = 'Lion'
        elif 'lynx' in text_lower:
            result['type'] = 'wildlifeSmall'
            result['name'] = 'Lynx'
        elif 'moose' in text_lower:
            result['type'] = 'wildlifeSmall'
            result['name'] = 'Moose'

        # Players
        elif 'power' in text_lower and 'scout' in text_lower:
            result['type'] = 'playerBad'
        elif 'power' in text_lower and 'reinforce' in text_lower:
            result['type'] = 'playerGood'

        # Extract level
        if 'lv.' in text_lower or 'lv ' in text_lower:
            parts = text.split('Lv')
            if len(parts) > 1:
                level_str = parts[1].split()[0].replace('.', '').strip()
                try:
                    result['level'] = int(level_str)
                except:
                    pass

        # Extract power
        if 'power' in text_lower:
            lines = text.split('\n')
            for line in lines:
                if 'power' in line.lower() and 'recommended' not in line.lower():
                    numbers = ''.join(filter(lambda x: x.isdigit() or x == ',', line))
                    if numbers:
                        try:
                            result['power'] = int(numbers.replace(',', ''))
                        except:
                            pass
                    break

        return result

    def scan_coordinate(self, x, y):
        """Scan single coordinate"""
        if not self.is_running:
            return None

        if self.should_skip_coordinate(x, y):
            self.skip_count += 1
            return None

        self.enter_coordinates(x, y)
        popup_text = self.click_center_and_read_popup()
        location_data = self.parse_popup_text(popup_text)
        location_data['coordinates'] = (x, y)

        cord_key = f"{x},{y}"
        self.map_data[cord_key] = location_data
        self.close_popup()

        self.scan_count += 1

        if location_data['type'] in self.multi_tile_objects:
            self.add_adjacent_skips(x, y, location_data['type'])

        return location_data

    def scan_area(self, start_x, start_y, end_x, end_y, step=1, progress_callback=None):
        """Scan area with progress callback"""
        self.is_running = True
        total = ((end_x - start_x) // step + 1) * ((end_y - start_y) // step + 1)

        for x in range(start_x, end_x + 1, step):
            for y in range(start_y, end_y + 1, step):
                if not self.is_running:
                    return

                self.scan_coordinate(x, y)

                if progress_callback:
                    progress = (self.scan_count + self.skip_count) / total * 100
                    progress_callback(progress, self.scan_count, self.skip_count)

    def stop(self):
        """Stop scanning"""
        self.is_running = False

    def save_data(self, filename):
        """Save map data"""
        with open(filename, 'w') as f:
            json.dump(self.map_data, f, indent=2)