import time
import numpy as np
import pyautogui
from PIL import Image
import cv2

from SourceCode.BaseClass import BaseClass


class FastScreenshotScanner(BaseClass):
    """Fast scanner that captures zoomed-out map screenshots and stitches them"""

    def __init__(self, config):
        super().__init__(config)

    def zoom_out(self):
        """Zoom out map to correct level"""
        if 'zoom_button' not in self.config:
            # Use F5 key to zoom out if no button configured
            pyautogui.press('f5', presses=16, interval=0.1)
            time.sleep(0.5)
            return

        zoom_clicks = self.config.get('zoom_clicks', 5)
        zoom_btn = self._percent_to_pixel(*self.config['zoom_button'])

        for _ in range(zoom_clicks):
            pyautogui.click(*zoom_btn)
            time.sleep(0.2)

        time.sleep(1)  # Wait for zoom to stabilize

    def reset_zoom(self):
        """Resets zoom button by entering then exiting the Town"""
        th_button = self.config.get('town_hall_button')
        th_adjusted_btn = self._percent_to_pixel(*th_button)

        pyautogui.press(*th_adjusted_btn, presses=2, interval=0.1)

        time.sleep(0.5)


    def capture_map_screenshot(self, x, y):
        """Capture screenshot of current map view"""
        region = self._percent_region_to_pixel(self.config['map_region'])
        screenshot = pyautogui.screenshot(region=region)

        # Store with metadata
        return {
            'image': screenshot,
            'coord': (x, y),
            'region': region
        }

    def scan_map_area(self, start_x, start_y, end_x, end_y, progress_callback=None):
        """Scan map area using screenshot method"""
        self.is_running = True
        self.screenshots = []

        step = self.config.get('coord_step', 50)
        coords_to_scan = []

        for x in range(start_x, end_x + 1, step):
            for y in range(start_y, end_y + 1, step):
                coords_to_scan.append((x, y))

        total = len(coords_to_scan)

        for idx, (x, y) in enumerate(coords_to_scan):
            if not self.is_running:
                break

            # Reset Zoom after each navigation
            self.reset_zoom()

            # Navigate to coordinate
            self.enter_coordinates(x, y)

            # Zoom out after each navigation
            self.zoom_out()

            # Capture screenshot
            screenshot_data = self.capture_map_screenshot(x, y)
            self.screenshots.append(screenshot_data)

            if progress_callback:
                progress = (idx + 1) / total * 100
                progress_callback(progress, idx + 1, total)

            time.sleep(0.3)

        return self.screenshots

    def stitch_screenshots(self, output_file='map_stitched.png', progress_callback=None):
        """Stitch screenshots together into one large image"""
        if not self.screenshots:
            return None

        cords = [(s['coord'][0], s['coord'][1]) for s in self.screenshots]
        min_x = min(c[0] for c in cords)
        max_x = max(c[0] for c in  cords)
        min_y = min(c[1] for c in cords)
        max_y = max(c[1] for c in cords)

        step = self.config.get('coord_step', 50)

        grid_w = (max_x - min_x) // step + 1
        grid_h = (max_y - min_y) // step + 1

        sample_img = np.array(self.screenshots[0]['image'])
        img_h, img_w = sample_img.shape[:2]

        canvas_w = grid_w * img_w
        canvas_h = grid_h * img_h

        print(f"Creating stitched image: {canvas_w}x{canvas_h} pixels")
        canvas = np.zeros((canvas_h, canvas_w, 3), dtype=np.uint8)

        for idx, screenshot_data in enumerate(self.screenshots):
            x, y = screenshot_data['coord']
            img = np.array(screenshot_data['image'])

            # Calculate grid position
            grid_x = (x - min_x) // step
            # FLIP Y-AXIS: game cords increase upward, image cords increase downward
            grid_y = (max_y - y) // step

            px = grid_x * img_w
            py = grid_y * img_h

            try:
                canvas[py:py + img_h, px:px + img_w] = img
            except:
                print(f"Warning: Could not place image at ({x}, {y})")

            if progress_callback:
                progress = (idx + 1) / len(self.screenshots) * 100
                progress_callback(progress)

        stitched = Image.fromarray(canvas)
        stitched.save(output_file)
        print(f"Stitched image saved to {output_file}")

        return output_file

    def stitch_screenshots_advanced(self, output_file='map_stitched_advanced.png'):
        """Advanced stitching using OpenCV feature matching"""

        if not self.screenshots or len(self.screenshots) < 2:
            return self.stitch_screenshots(output_file)  # Fall back to simple stitching

        # Sort screenshots by coordinates
        sorted_shots = sorted(self.screenshots, key=lambda s: (s['coord'][1], s['coord'][0]))

        # Use OpenCV's Stitcher
        stitcher = cv2.Stitcher_create(cv2.Stitcher_PANORAMA)

        images = [np.array(s['image']) for s in sorted_shots]

        print("Performing advanced stitching with feature matching...")
        status, stitched = stitcher.stitch(images)

        if status == cv2.Stitcher_OK:
            # Convert BGR to RGB
            stitched_rgb = cv2.cvtColor(stitched, cv2.COLOR_BGR2RGB)
            result = Image.fromarray(stitched_rgb)
            result.save(output_file)
            print(f"Advanced stitched image saved to {output_file}")
            return output_file
        else:
            print(f"Advanced stitching failed (status={status}), falling back to grid stitching")
            return self.stitch_screenshots(output_file)

    def stop(self):
        """Stop scanning"""
        self.is_running = False
