import json
import os
import sys
import pyautogui
import tkinter as tk
from tkinter import ttk, messagebox, filedialog

from SourceCode.PositionCapture import PositionCapture
from SourceCode.ScannerGui import ScannerGUI


class ConfigWizard():
    """Setup wizard for initial configuration"""

    def __init__(self):
        self.config = {}
        self.root = tk.Tk()
        self.root.title("Game Scanner Setup Wizard")
        self.root.geometry("600x500")

        # Set window icon
        self._set_icon()

    def _set_icon(self):
        """Set the window icon"""
        try:
            if getattr(sys, 'frozen', False):
                application_path = os.path.dirname(sys.executable)
            else:
                application_path = os.path.dirname(os.path.abspath(__file__))

            icon_path = os.path.join(application_path, 'icon.ico')
            if os.path.exists(icon_path):
                self.root.iconbitmap(icon_path)
        except:
            pass  # Icon not found or invalid

    def run(self):
        welcome = """Welcome to Game Map Scanner Setup!

This wizard will help you configure the scanner for your screen.

You'll need to:
1. Identify screen positions for coordinate entry
2. Define the popup region for OCR reading
3. Configure scanning preferences

Make sure the game is open and visible before proceeding."""

        frame = ttk.Frame(self.root, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        ttk.Label(frame, text=welcome, justify=tk.LEFT).pack(pady=20)

        ttk.Button(frame, text="Start Setup",
                   command=self.start_setup).pack(pady=10)
        ttk.Button(frame, text="Load Existing Config",
                   command=self.load_config).pack(pady=10)
        ttk.Button(frame, text="Exit",
                   command=self.root.destroy).pack(pady=10)

        self.root.mainloop()
        return self.config

    def start_setup(self):
        # Hide the main wizard window during capture
        self.root.withdraw()

        steps = [
            ("Coordinate Screen Button", "coordinate_screen"),
            ("X Input Field", "x_input"),
            ("Y Input Field", "y_input"),
            ("Confirm Button", "confirm_button"),
            ("Center Click Position", "center_click"),
            ("Close Popup Position", "close_popup"),
        ]

        for label, key in steps:
            pos = PositionCapture.capture_position(label)
            if pos is None:
                self.root.deiconify()
                messagebox.showwarning("Cancelled", "Setup cancelled")
                return
            self.config[key] = pos

        # Capture popup region
        region = PositionCapture.capture_region("Popup Region")
        if region is None:
            self.root.deiconify()
            messagebox.showwarning("Cancelled", "Setup cancelled")
            return
        self.config['popup_region'] = region

        # Show main window again
        self.root.deiconify()

        # Ask about fast scanner mode setup
        if messagebox.askyesno("Fast Scanner", "Do you want to configure Fast Screenshot Scanner mode?"):
            self.root.withdraw()

            # Capture zoom slider position
            zoom_pos = PositionCapture.capture_position("Zoom Out Button (click to zoom out fully)")
            if zoom_pos:
                self.config['zoom_button'] = zoom_pos

            # Capture TownHall button position
            town_hall_button_pos = PositionCapture.capture_position("Town Hall Button Position")
            if town_hall_button_pos:
                self.config['town_hall_button'] = town_hall_button_pos

            # Capture map view region
            map_region = PositionCapture.capture_region("Map View Area (visible map when zoomed out)")
            if map_region:
                self.config['map_region'] = map_region

            self.root.deiconify()

            # Get zoom level settings
            zoom_dialog = tk.Toplevel(self.root)
            zoom_dialog.title("Zoom Settings")
            zoom_dialog.geometry("600x500")
            zoom_dialog.attributes('-topmost', True)

            ttk.Label(zoom_dialog, text="Number of zoom-out clicks:").pack(pady=10)
            zoom_clicks = ttk.Entry(zoom_dialog)
            zoom_clicks.insert(0, "5")
            zoom_clicks.pack()

            ttk.Label(zoom_dialog, text="Coordinate step size (tiles per screenshot):").pack(pady=10)
            coord_step = ttk.Entry(zoom_dialog)
            coord_step.insert(0, "50")
            coord_step.pack()

            result = {'done': False}

            def save_zoom():
                self.config['zoom_clicks'] = int(zoom_clicks.get())
                self.config['coord_step'] = int(coord_step.get())
                result['done'] = True
                zoom_dialog.destroy()

            ttk.Button(zoom_dialog, text="Save", command=save_zoom).pack(pady=20)
            zoom_dialog.wait_window()

        # Save config
        filename = filedialog.asksaveasfilename(
            defaultextension=".json",
            filetypes=[("JSON files", "*.json")],
            initialfile="scanner_config.json"
        )

        if filename:
            with open(filename, 'w') as f:
                json.dump(self.config, f, indent=2)
            messagebox.showinfo("Success", f"Configuration saved to {filename}")
            self.root.destroy()

    def load_config(self):
        filename = filedialog.askopenfilename(
            filetypes=[("JSON files", "*.json")]
        )

        if filename:
            with open(filename, 'r') as f:
                self.config = json.load(f)
            messagebox.showinfo("Success", "Configuration loaded")
            self.root.destroy()


def main():
    # Run setup wizard first
    wizard = ConfigWizard()
    config = wizard.run()

    if not config:
        return

    # Launch main application
    app = ScannerGUI(config)
    app.run()


if __name__ == "__main__":
    main()